/**
 ******************************************************************************
 * @file    drv_uart.h 
 * @author  zhu_yanjun@superstunning.com
 * @date    2017/01/10
 * @brief   uart driver
 ******************************************************************************
 * @attention
 *
 *
 ******************************************************************************
 */

#include "drv_uart.h"
#include "string.h"

/**
 * usart object
 */
usart_t usart1;


//typedef struct
//{
//	bool flg_msg;
//	drv_message_t message;
//	drv_message_t message_buffer;
//}usart_message_t;

//usart_message_t usart1_message;


static drv_message_t* message;
static drv_message_t message_buffer;


/**
 * @function uart_config
 * @brief    system tick initialization PA9(TX),PA10(RX)
 * @param    none
 * @retval   none
 */
static void usart1_config(uint32_t baudrate)
{
		
	USART_InitTypeDef USART_InitStructure; 
	NVIC_InitTypeDef NVIC_InitStructure;
	USART_ClockInitTypeDef USART_ClockInitStructure;
	GPIO_InitTypeDef GPIO_InitStructure;
	
	
	// enable uart1 clock
	RCC_APB2PeriphClockCmd(RCC_APB2Periph_USART1,ENABLE);
	
	
	// UART1 - GPIO - TX
	GPIO_InitStructure.GPIO_Pin = GPIO_Pin_9;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;  	
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;  
	GPIO_Init(GPIOA, &GPIO_InitStructure);
	
	// UART1 - GPIO - RX
	GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;  	
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;  
	GPIO_Init(GPIOA, &GPIO_InitStructure);
	
	
	
	
	
	// UART1 - baudrate configuration
	USART_InitStructure.USART_BaudRate = baudrate;
	USART_InitStructure.USART_WordLength = USART_WordLength_8b;
	USART_InitStructure.USART_StopBits = USART_StopBits_1;
	USART_InitStructure.USART_Parity = USART_Parity_No;
	USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None; 
	USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx; 
	USART_Init(USART1, &USART_InitStructure);
	
	// UART1 clock configuration
	USART_ClockInitStructure.USART_Clock = USART_Clock_Disable;
	USART_ClockInitStructure.USART_CPOL = USART_CPOL_Low;
	USART_ClockInitStructure.USART_CPHA = USART_CPHA_2Edge;
	USART_ClockInitStructure.USART_LastBit = USART_LastBit_Disable;
	USART_ClockInit(USART1,&USART_ClockInitStructure);
	
	// UART1 interrupt configuration
	NVIC_InitStructure.NVIC_IRQChannel = USART1_IRQn;
	NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 2;
	NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0;
	NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
	NVIC_Init(&NVIC_InitStructure);  
	USART_ITConfig(USART1, USART_IT_RXNE, ENABLE);
	
	// enable uart1
	USART_Cmd(USART1, ENABLE);
		
		
		
		
		
		
		
		
		
		
		
}

/**
 * @function usart1_send_byte
 * @brief    send byte by usart
 * @param    data - data to send
 * @retval   none
 */
static void usart1_send_byte(uint8_t data)
{
	USART_SendData(USART1, data);

	while (USART_GetFlagStatus(USART1, USART_FLAG_TC) == RESET)
	{}
}

/**
 * @function usart1_send_data
 * @brief    send data by usart
 * @param    data - data array to send
 * @param    size - data array size
 * @retval   none
 */
static void usart1_send_data(uint8_t* data, uint16_t size)
{
	uint16_t cnt = size;
	uint8_t* ptr = data;
	
	if (ptr)
	{
		for(;cnt > 0; cnt--, ptr++)
		{
			usart1_send_byte(*ptr);
		}
	}
}

/**
 * @function usart1_send_string
 * @brief    send string by usart
 * @param    str - string to send
 * @retval   none
 */
static void usart1_send_string(char* str)
{
	uint8_t* ptr = (uint8_t*)str;
	
	if (ptr)
	{
		for(;*ptr != '\0'; ptr++)
		{
			usart1_send_byte(*ptr);
		}
	}
}

/**
 * @function usart1_run
 * @brief    usart1 process
 * @param    str - string to send
 * @retval   none
 */
static void usart1_run(void)
{
	if (message)
	{
		if (usart1.message_handler)
		{
			usart1.message_handler(message);
		}
		
		message = null;
	}
}


/**
 * @function usart1_handler
 * @brief    usart1 irq handler
 * @param    none
 * @retval   none
 */
void usart1_handler(void)
{
	if(USART_GetITStatus(USART1, USART_IT_RXNE) != RESET)
	{
		uint8_t data;
		// clear interrupt flag
		USART_ClearITPendingBit(USART1, USART_IT_RXNE);
		/* Read one byte from the receive data register */
		data = USART_ReceiveData(USART1);
		//
		if (usart1.message_parser)
		{
			drv_message_t* msg = usart1.message_parser(data);
			
			if (msg)
			{
				memcpy(message_buffer.message, msg->message, msg->size);
				message_buffer.size = msg->size;
				message = &message_buffer;
				msg->size = 0;
			}
			
		}
	}
}

/**
 * @function usart_init
 * @brief    usart init for all usart port
 * @param    none
 * @retval   none
 */
void usart_init(void)
{
	usart1.config = usart1_config;
	usart1.message_handler = null;
	usart1.message_parser = null;
	usart1.send_byte = usart1_send_byte;
	usart1.send_data = usart1_send_data;
	usart1.send_string = usart1_send_string;
	usart1.run = usart1_run;
	
	message = null;
	message_buffer.size = 0;
	
}

/**
 * @function usart_set_message_parser
 * @brief    set usart message parser
 * @param    usart - usart object
 * @param    parser - message parser function
 * @retval   none
 */
void usart_set_message_parser(usart_t* usart, drv_message_t* (*parser)(uint8_t data))
{
	usart->message_parser = parser;
}

/**
 * @function usart_set_message_handler
 * @brief    set usart message handler
 * @param    usart - usart object
 * @param    handler - message handler function
 * @retval   none
 */
void usart_set_message_handler(usart_t* usart, void (*handler)(drv_message_t* message))
{
	usart->message_handler = handler;
}
